package es.uniovi.healthappv2;

import android.app.Service;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothGattService;
import android.bluetooth.BluetoothManager;
import android.bluetooth.BluetoothProfile;
import android.content.Context;
import android.content.Intent;
import android.os.Binder;
import android.os.IBinder;
import android.util.Log;

import java.nio.ByteBuffer;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.SecretKeySpec;

public class BluetoothLeService extends Service {
    private final static String TAG = BluetoothLeService.class.getSimpleName();

    private BluetoothManager bluetoothManager;
    private BluetoothAdapter bluetoothAdapter;
    private String bluetoothDeviceAddress;
    private BluetoothGatt bluetoothGatt;

    private final IBinder iBinder = new LocalBinder();

    private int connectionState = STATE_DISCONNECTED;

    private static final int STATE_DISCONNECTED = 0;
    private static final int STATE_CONNECTING = 1;
    private static final int STATE_CONNECTED = 2;

    public final static String ACTION_GATT_CONNECTED =
            "es.uniovi.myapplication.ACTION_GATT_CONNECTED";
    public final static String ACTION_GATT_DISCONNECTED =
            "es.uniovi.myapplication.ACTION_GATT_DISCONNECTED";
    public final static String ACTION_GATT_SERVICES_DISCOVERED =
            "es.uniovi.myapplication.ACTION_GATT_SERVICES_DISCOVERED";
    public final static String ACTION_DATA_AVAILABLE =
            "es.uniovi.myapplication.ACTION_DATA_AVAILABLE";
    public final static String EXTRA_DATA =
            "es.uniovi.myapplication.EXTRA_DATA";

    public final static UUID UUID_HEART_RATE_MEASUREMENT =
            UUID.fromString(SampleGattAttributes.HEART_RATE_MEASUREMENT);



    // Se usará para conectarse al servidor Gatt
    private final BluetoothGattCallback gattCallback =
            new BluetoothGattCallback() {
                @Override
                public void onConnectionStateChange(BluetoothGatt gatt, int status,
                                                    int newState) {
                    Log.d("ConnectionStateChange()","Inicio");
                    String intentAction;
                    if (newState == BluetoothProfile.STATE_CONNECTED) {
                        intentAction = ACTION_GATT_CONNECTED;
                        connectionState = STATE_CONNECTED;
                        broadcastUpdate(intentAction);
                        Log.i(TAG, "Connected to GATT server.");
                        Log.i(TAG, "Attempting to start service discovery:" +
                                bluetoothGatt.discoverServices());

                    } else if (newState == BluetoothProfile.STATE_DISCONNECTED) {
                        intentAction = ACTION_GATT_DISCONNECTED;
                        connectionState = STATE_DISCONNECTED;
                        Log.i(TAG, "Disconnectedd from GATT server.");

                        broadcastUpdate(intentAction);
                    }
                }

                @Override

                public void onServicesDiscovered(BluetoothGatt gatt, int status) {
                    if (status == BluetoothGatt.GATT_SUCCESS) {
                        broadcastUpdate(ACTION_GATT_SERVICES_DISCOVERED);
                    } else {
                        Log.w(TAG, "onServicesDiscovered received: " + status);
                    }
                }

                @Override

                public void onCharacteristicRead(BluetoothGatt gatt,
                                                 BluetoothGattCharacteristic characteristic,
                                                 int status) {
                    if (status == BluetoothGatt.GATT_SUCCESS) {
                        Log.d("onCharcateristicRead()", "Se llama a broadcastUpdate(2)");
                        broadcastUpdate(ACTION_DATA_AVAILABLE, characteristic);
                    }
                    else{
                        Log.d("onCharacteristicRead()", "Not sucess GATT"+status);
                    }
                }

                @Override
                public void onCharacteristicChanged(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic) {
                    Log.d(TAG, "onCharacteristicChanged() ejecutado. Se ha modificado el valor del ritmo cardiaco");
                    broadcastUpdate(ACTION_DATA_AVAILABLE, characteristic);
                }
            };

    //Metodo llamado desde DeviceControlActivity para inicializar el manager y adapter
    public boolean initialize() {
        Log.d("initialize()", "inicio.");
        if (bluetoothManager == null) {
            bluetoothManager = (BluetoothManager) getSystemService(Context.BLUETOOTH_SERVICE);
            if (bluetoothManager == null) {
                Log.d(TAG, "Error al inicializar BluetoothManager.");
                return false;
            }
        }
        bluetoothAdapter = bluetoothManager.getAdapter();
        if (bluetoothAdapter == null) {
            Log.d(TAG, "No se pudo obtener BluetoothAdapter.");
            return false;
        }
        Log.d("initialize()", "retorna true.");
        return true;
    }

    //Metodo para realizar la conexion con el servidor GATT (pulsera)
    public boolean connect(final String address) {
        Log.d("connect()", "Inicio de la conexión con el servidor GATT");
        if (bluetoothAdapter == null || address == null) {
            Log.d("connect()", "El adapter no ha sido inicializado o la direccion no es correcta");
            return false;
        }

        // Intento de reconexion a un dispositivo anterior
        if (bluetoothDeviceAddress != null && address.equals(bluetoothDeviceAddress)
                && bluetoothGatt != null) {
            Log.d("connect()", "Intento de reconexion a un dispositivo anterior");
            if (bluetoothGatt.connect()) {
                connectionState = STATE_CONNECTING;
                Log.d("connect()", "ret true");
                return true;
            } else {
                Log.d("connect()", "ret false");
                return false;
            }
        }

        final BluetoothDevice device = bluetoothAdapter.getRemoteDevice(address);
        if (device == null) {
            Log.d("connect()", "No se ha encontrado el dispositivo. No se puede conectar");
            return false;
        }
        Log.d("connect()", "Se ha encontrado el dispositivo. El intento de conexion sigue en curso");

        //Se realiza la conexión
        bluetoothGatt = device.connectGatt(this, true, gattCallback);
        Log.d("connect()", "Realizando la conexión..."+bluetoothGatt.getDevice()+"--");
        bluetoothDeviceAddress = address;
        connectionState = STATE_CONNECTING;
        return true;
    }

    public void disconnect() {
        if (bluetoothAdapter == null || bluetoothGatt == null) {
            Log.w("connect()", "BluetoothAdapter no fue inicualizado");
            return;
        }
        bluetoothGatt.disconnect();
    }

    //Método para leer una característica que se pasa como parametro.
    public boolean readCharacteristic(BluetoothGattCharacteristic characteristic) {
        if (bluetoothAdapter == null || bluetoothGatt == null) {
            Log.w(TAG, "BluetoothAdapter not initialized");
            return false;
        }
        Log.d("readCharacteristic()", "Se invoca al metodo del bluetoothGatt");
        return bluetoothGatt.readCharacteristic(characteristic);
    }

    //Método para escribir una característica que se pasa como parametro.
    public boolean writeCharacteristic(BluetoothGattCharacteristic characteristic, byte[] value) {
        if (bluetoothAdapter == null || bluetoothGatt == null) {
            Log.w(TAG, "BluetoothAdapter not initialized");
            return false;
        }
        Log.d("writeCharacteristic()", "Se invoca al metodo del bluetoothGatt");
        characteristic.setValue(value);

        return bluetoothGatt.writeCharacteristic(characteristic);
    }

    private void broadcastUpdate(final String action) {
            final Intent intent = new Intent(action);
            Log.d("broadcastUpdate()", "action: "+action);
            sendBroadcast(intent);
    }

    private void broadcastUpdate(final String action,
                                 final BluetoothGattCharacteristic characteristic) {
        final Intent intent = new Intent(action);
        Log.d("broadcastUpdate()", "INICIO");

        // Para obtener el formato de los datos se utiliza la siguiente especificación:
        // https://www.bluetooth.com/wp-content/uploads/Sitecore-Media-Library/Gatt/Xml/Characteristics/org.bluetooth.characteristic.heart_rate_measurement.xml

        //Si la caracteristica coincide con ritmo cardiaco:
        if (UUID_HEART_RATE_MEASUREMENT.equals(characteristic.getUuid())) {
            Log.d("broadcastUpdate()", "La caracteristica es ritmo carciaco");
            int flag = characteristic.getProperties();
            int format = -1;
            if ((flag & 0x01) != 0) {
                format = BluetoothGattCharacteristic.FORMAT_UINT16;
                Log.d("broadcastUpdate()", "Formato UINT16.");
            } else {
                format = BluetoothGattCharacteristic.FORMAT_UINT8;
                Log.d("broadcastUpdate()", "Formato UINT8.");
            }
            final int heartRate = characteristic.getIntValue(format, 1);
            Log.d("broadcastUpdate()", String.format("Recibido ritmo cardiaco: %d", heartRate));
            intent.putExtra(EXTRA_DATA, String.valueOf(heartRate));
        }
        //Si la característica coincide con datos de actividad(pasos, distancia, calorías)
        else if(SampleGattAttributes.UUID_STEPS.equals(characteristic.getUuid())){
            Log.d("broadcastUpdate()","La caracteristica es steps");

            final byte[] data = characteristic.getValue();
            //https://blog.infullmobile.com/introduction-to-bluetooth-le-on-ios-mi-band-2-case-study-343153921877
            final byte[] data_steps={0x00,0x00,0x00,0x00,data[4],data[3],data[2],data[1]};
            Log.d("broadcastUpdat(steps_t)", String.format("data[0]=0x%02X",data[0]));
            long pasos=bytesToLong(data_steps);
            final byte[] data_intens={0x00,0x00,0x00,0x00,data[8],data[7],data[6],data[5]};
            long intens=bytesToLong(data_intens);
            final byte[] data_act={0x00,0x00,0x00,0x00,data[12],data[11],data[10],data[9]};
            long act=bytesToLong(data_act);

            Log.d("broadcastUpdate(steps)", String.format("Recibidos pasos: %d", pasos));
            Log.d("broadcastUpdate(steps)", String.valueOf(pasos));
            intent.putExtra(EXTRA_DATA, "SPasos: "+String.valueOf(pasos)+" Distancia(m): "+intens+" Calorías: "+act);
        }

        //Otro tipo de característica
        else {
            Log.d("broadcastUpdate()","Característica general");

            // For all other profiles, writes the data formatted in HEX.
            final byte[] data = characteristic.getValue();
            if (data != null && data.length > 0) {
                final StringBuilder stringBuilder = new StringBuilder(data.length);
                for(byte byteChar : data)
                    stringBuilder.append(String.format("%02X ", byteChar));
                intent.putExtra(EXTRA_DATA, new String(data) + "\n" + stringBuilder.toString());
            }
        }
        //DATA_AVAILABLE
        sendBroadcast(intent);
    }

    public void pasos(){
        BluetoothGattCharacteristic c=bluetoothGatt.getService(SampleGattAttributes.CUSTOM_SERVICE_FEE0).getCharacteristic(SampleGattAttributes.UUID_STEPS);
        this.readCharacteristic(c);
    }



    public void setCharacteristicNotification(BluetoothGattCharacteristic characteristic,
                                              boolean enabled) {
        if (bluetoothAdapter == null || bluetoothGatt == null) {
            Log.w(TAG, "BluetoothAdapter not initialized");
            return;
        }
        bluetoothGatt.setCharacteristicNotification(characteristic, enabled);

        //Operación específica para el sensor de ritmo cardiaco
        if (UUID_HEART_RATE_MEASUREMENT.equals(characteristic.getUuid())) {
            BluetoothGattDescriptor descriptor = characteristic.getDescriptor(
                    UUID.fromString(SampleGattAttributes.UUID_CLIENT_CHARACTERISTIC_CONFIG));
            descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
            bluetoothGatt.writeDescriptor(descriptor);
        }
    }
    //Metodo para obtener los servicios soportados por el dispositivo
    public List<BluetoothGattService> getSupportedGattServices() {
        if (bluetoothGatt == null) return null;
        return bluetoothGatt.getServices();
    }

    public long bytesToLong(byte[] bytes) {
        return ByteBuffer.wrap(bytes).getLong();
    }


    public class LocalBinder extends Binder {
        public BluetoothLeService getService() {
            return BluetoothLeService.this;
        }
    }

    @Override
    public IBinder onBind(Intent intent) {
        Log.d(TAG, "Metodo onBind de BtLeService");
        return iBinder;
    }
}
