#include "ComputeFunctions.h"


/* Para medir tiempos */
double Ctimer(void) {
  struct timeval tm;

  gettimeofday(&tm, NULL);

  return tm.tv_sec + tm.tv_usec/1.0E6;
}


/*  Returns the next power of 2 of a given number */
long nextPow2(int x)
{
   --x;
   x |= x >> 1;
   x |= x >> 2;
   x |= x >> 4;
   x |= x >> 8;
   x |= x >> 16;
   return ++x;
}


/* Funcion de comparacion para ordenar desendentemente con qsort */
int comparadesc(const void *a, const void *b)
{
  ValueAndPos *A=(ValueAndPos *)a;  
  ValueAndPos *B=(ValueAndPos *)b; 
   
  return (B->data - A->data);
}


/* Funcion de comparacion para ordenar ascendentemete con qsort */
int compara(const void *a, const void *b)
{
  return *((int*)a)-*((int*)b);
}


/* Funcion que retorna la PRIMERA posicion de V de valor igual a val */
int buscarFirstIgual(const int n, const MyType *V, const MyType val)
{
  int i=0, pos=-1;
  bool noEncontrado=true;
  
  while (i<n && noEncontrado)
  {
    if (V[i]==val) { pos=i; noEncontrado=false; }
    i++;
  }
  return pos;
}


/* Funcion que retorna la PRIMERA posicion de V de valor mayor que val */
int buscarFirstMayor(const int n, const MyType *V, const MyType val)
{
  int i=0, pos=-1;
  bool noEncontrado=true;
  
  while (i<n && noEncontrado)
  {
    if (V[i]>val) { pos=i; noEncontrado=false; }
    i++;
  }
  return pos;
}


/* Funcion que retorna la PRIMERA posicion de V de valor mayor o igual que val */
int buscarFirstMayorIgual(const int n, const MyType *V, const MyType val)
{
  int i=0, pos=-1;
  bool noEncontrado=true;
  
  while (i<n && noEncontrado)
  {
    if (V[i]>=val) { pos=i; noEncontrado=false; }
    i++;
  }
  return pos;
}


/* Funcion que retorna la ULTIMA posicion de V de valor menor o igual que val */
int buscarLastMenorIgual(const int n, const MyType *V, const MyType val)
{
  int i=0;

  while ((V[i]<val) && (i<n)) { i++; }

  return (i-1);
}


/* En el mismo fichero que la usa ¿?, de lo contrario da un warning, que ... */
inline void memSetValue(const int, const MyType, MyType *);

/* "#pragma GCC ivdep" mismo efecto que "MyType *__restrict". Gcc 4.x no soporta   */
/* "#pragma GCC ivdep" por lo que en este caso se debe usar "MyType *__restrict"   */
/* Por claridad usamos el pragma. Con OpenMP no se puede usar a la vez con "pragma */
/* parallel for", usar "#pragma omp simd" o "#pragma omp parallel for simd"        */
inline void memSetValue(const int n, const MyType value, MyType *v)
{
  int i; 

  #pragma GCC ivdep
  for(i=0; i<n; i++)  // openmp overhead > saved time ==> NO
    v[i]=value;
}


void memSetValueRealComplex(const int n, const MyType value, MyFFTcompl *v)
{
  int i; 

  #pragma GCC ivdep
  for(i=0; i<n; i++)  // openmp overhead > saved time ==> NO
    v[i]=value + 0.0*I;
}


void suma(const int n, MyType *dest, const MyType *src)
{
  int i;
  #pragma GCC ivdep
  for(i=0; i<n; i++)  // openmp overhead > saved time ==> NO
    dest[i]+=src[i];
}


/* Calcula la autocorrelation (c) del vector x. Se cumple que la dimension de corr, */
/* que es "n), es la de x, que es "m", por 2 menos 1. Esto es n=2m-1                */
void oldXcorr(const int n, MyType *corr, const int m, const MyType *x)
{
  int i, j;
  MyType dtmp;

  #ifdef OMP
    #pragma omp parallel for private(dtmp,j)
  #endif
  for(i=0; i<m; i++)
  {
    dtmp=0.0;
    for(j=0; j<(m-i); j++)
      dtmp += x[j] * x[j+i];
    corr[m+i-1]=corr[m-i-1]=dtmp;
  }
}


/* Regresion Lineal Simple = calcular b0 y la pendiente (slope) de la recta que ... */
/* No da los mismos resultados que Matlab ¿el metodo? Da igualn no se esta usando   */
void simpleRL(const int n, const MyType *Y, MyType *b0, MyType *slope)
{
  /* La formula, para eje ordenadas (X) y el de abscisas (Y) con n terminos es        */
  /* slope = (n*sum(X(i)*Y(i)) - sum(X(i))*sum(Y(i))) / (n*sum(X(i)^2) - sum(X(i))^2) */
  /* b0    = (sum(Y(i)) - slope*sum(X(i))) / n                                        */
  /* En nuestro problema X(i)=i, esto es, x=[1, 2, 3, 4,..., n}, por lo que podemos   */
  /* sum(X(i))  = (n(n+1)) / 2                                                        */
  /* sum(X(i)^2)= (n(n+1)(2n+1)) / 6 = (sum(X(i))(2n+1)) / 3                          */
  /* En nuestro problema Y(i)>=0, para todo i por lo que podemos usar cblas_?adsum    */
  /* En caso de no ser >=0 y querer blas podemos usar ?dot sin recurrir a usar otro   */
  /* vector. Sea nuestro vector Z y su tamano M, entonces:                            */
  /* double/float K=1.0;                                                              */
  /* ?dot(M, Z, 1, &K, 0)                                                             */
  /* Usar BLAS aqui no es mejor porque necesita 2 llamadas y en la version a mano con */
  /* un for se pueden calcular a la vez el sumY y el sumXY. Si n fuese mas grande ¿?¿ */
  int i;
  MyType sumX, sumX2, sumY=0, sumXY=0;

  sumX =n*(1.0+n)/2.0;
  sumX2=sumX*(2.0*n+1.0)/3.0;

  /* Version hand-made */
  #ifdef OMP2
    #pragma omp parallel for reduction(+:sumY) reduction(+:sumXY) // ¿openmp overhead > saved time?
  #endif
  for(i=0;i<n;i++) { sumY+=Y[i]; sumXY+=i*Y[i]; }

  /* Version usando BLAS */
  /*#ifdef SIMPLE
    sumY =cblas_sasum(n, Y, 1);
    sumXY=cblas_sdot (n, X, 1, Y, 1);
  #else
    sumY =cblas_dasum(n, Y, 1);
    sumXY=cblas_ddot (n, X, 1, Y, 1);
  #endif*/

  *slope=(n*sumXY - sumX*sumY) / (n*sumX2 - sumX*sumX);
  *b0   =(sumY-(*slope)*sumX)  / n;
}


/* This hanning initialization is the same as in Matlab R1019b */
int hanning(const int n, MyType *v)
{
  int
    i, Half,
    SMOne=n-1;
   
  MyType
    tmp=DosPI/(MyType)(n+1);

  if ((n%2)==0) Half=n/2; else Half=(n+1)/2;

  #ifdef OMP2
    #pragma omp parallel for simd  // ¿overhead > saved time?
  #else
    #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
  #endif
  for(i=0; i<Half; i++) 
  {
    #ifdef SIMPLE
      v[i]=0.5*(1.0 - cosf(tmp*(i+1)));
    #else
      v[i]=0.5*(1.0 -  cos(tmp*(i+1)));
    #endif
    v[SMOne-i] = v[i];
  }
  return OK;
}


int cHannFFT(const int winSize, const int nFrames, const int NoOverlap, const MyType *audio, const MyType *vHanning,
             const int rowsNMF, MyType *rSNMF, MyFFTcompl *cSNMF, const int fftSize, const int maxThreads, const int planType)
{
  int          k;
  MyFFTcompl   *xFFT=NULL;
  MyFFTCPUType *planFFT=NULL;

  /* Prepare FFT. Se hacer in-place, usando el mismo vector para entrada que salida para  */
  /* ahorrar memoria (512*8*maxThreads bytes o 1024*8*maxThreads bytes dependiendo cuando */
  /* se llame.                                                                            */
  /*                                                                                      */
  /* Recurrimos a esta forma de paralelismo porque la compilacion con la version paralela */
  /* de FFTW esta arrojando tiempos malos, cosa que en ReMAS, etc. eran buenos ¿el tamano?*/
  /*                                                                                      */
  /* Al hacerlo asi no necesitamos una funcion especial HannFFT cuando winSize sea menor  */
  /* que fftSize, que es el caso de HSD                                                   */ 
  /*                                                                                      */
  /* Las funciones de la API de FFTW NO son thread-safe, salvo fft_execute(). Es por ello */
  /* que debemos declarar/liberar todo fuera del "pragma omp parallel". Es lo que hay     */
  /*                                                                                      */
  /* Si no se han creado los palnes wisdom (no exite fichero Planes.txt para la maquina,  */
  /* entonces comentar las llamadas a planFFT activas y des-comentar las comentadas       */
  #ifdef SIMPLE
    CHECKNULL(xFFT   =(MyFFTcompl   *)fftwf_malloc(sizeof(MyFFTcompl)  *fftSize*maxThreads));
    CHECKNULL(planFFT=(MyFFTCPUType *)fftwf_malloc(sizeof(MyFFTCPUType)*maxThreads)); 
  #else
    CHECKNULL(xFFT   =(MyFFTcompl   *)fftw_malloc(sizeof(MyFFTcompl)  *fftSize*maxThreads));
    CHECKNULL(planFFT=(MyFFTCPUType *)fftw_malloc(sizeof(MyFFTCPUType)*maxThreads)); 
  #endif
  for(k=0;k<maxThreads;k++)
  {
    #ifdef SIMPLE
      planFFT[k]=fftwf_plan_dft_1d(fftSize, &xFFT[k*fftSize], &xFFT[k*fftSize], FFTW_FORWARD, planType);
    #else
      planFFT[k]= fftw_plan_dft_1d(fftSize, &xFFT[k*fftSize], &xFFT[k*fftSize], FFTW_FORWARD, planType);
    #endif
  }

  #ifdef OMP
    #pragma omp parallel
  #endif
  {
    int myID, myIDpos, pos, i, j;

    #ifdef OMP
      myID=omp_get_thread_num();
    #else
      myID=0;
    #endif
    myIDpos=myID*fftSize;

    #ifdef OMP
      #pragma omp for
    #endif
    for(i=0; i<nFrames; i++)
    {
      pos=i*(winSize-NoOverlap);

      #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
      for(j=0; j<winSize; j++) { xFFT[myIDpos+j]=audio[pos+j] * vHanning[j]; }

      /* We need this for the cases where winsize < fftSize. We observed */
      /* issues with 7 seconds audio ¿why not with long audios? ?¿?¿     */
      if(winSize<fftSize)
        memset(&xFFT[myIDpos+winSize], 0, sizeof(MyFFTcompl)*(fftSize-winSize));

      #ifdef SIMPLE
        fftwf_execute(planFFT[myID]);
      #else
        fftw_execute(planFFT[myID]);
      #endif

      pos=i*rowsNMF;
      
      #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
      for(j=0; j<rowsNMF; j++)
      {
        #ifdef SIMPLE
          rSNMF[pos+j]=cabsf(xFFT[myIDpos+j]);
        #else
          rSNMF[pos+j]= cabs(xFFT[myIDpos+j]);
        #endif
        cSNMF[pos+j]=xFFT[myIDpos+j];
      }
    }
  }

  for(k=0;k<maxThreads;k++)
  {
    #ifdef SIMPLE
      fftwf_destroy_plan(planFFT[k]);
    #else
      fftw_destroy_plan(planFFT[k]);
    #endif
  }
  
  #ifdef SIMPLE
    fftwf_free(xFFT);
    fftwf_free(planFFT);
  #else
    fftw_free(xFFT);
    fftw_free(planFFT);
  #endif

  return OK;
}


int rHannFFT(const int winSize, const int nFrames, const int NoOverlap, const MyType *audio, const MyType *vHanning,
             const int rowsNMF, MyType *sNMF, const int fftSize, const int maxThreads, const int planType)
{
  int          k;
  MyFFTcompl   *xFFT=NULL;
  MyFFTCPUType *planFFT=NULL;

  /* Prepare FFT. Se hacer in-place, usando el mismo vector para entrada que salida para  */
  /* ahorrar memoria (512*8*maxThreads bytes o 1024*8*maxThreads bytes dependiendo cuando */
  /* se llame.                                                                            */
  /*                                                                                      */
  /* Recurrimos a esta forma de paralelismo porque la compilacion con la version paralela */
  /* de FFTW esta arrojando tiempos malos, cosa que en ReMAS, etc. eran buenos ¿el tamano?*/
  /*                                                                                      */
  /* Al hacerlo asi no necesitamos una funcion especial HannFFT cuando winSize sea menor  */
  /* que fftSize, que es el caso de HSD                                                   */ 
  /*                                                                                      */
  /* Las funciones de la API de FFTW NO son thread-safe, salvo fft_execute(). Es por ello */
  /* que debemos declarar/liberar todo fuera del "pragma omp parallel". Es lo que hay     */
  /*                                                                                      */
  /* Si no se han creado los palnes wisdom (no exite fichero Planes.txt para la maquina,  */
  /* entonces comentar las llamadas a planFFT activas y des-comentar las comentadas       */
  #ifdef SIMPLE
    CHECKNULL(xFFT   =(MyFFTcompl   *)fftwf_malloc(sizeof(MyFFTcompl)  *fftSize*maxThreads));
    CHECKNULL(planFFT=(MyFFTCPUType *)fftwf_malloc(sizeof(MyFFTCPUType)*maxThreads)); 
  #else
    CHECKNULL(xFFT   =(MyFFTcompl   *)fftw_malloc(sizeof(MyFFTcompl)  *fftSize*maxThreads));
    CHECKNULL(planFFT=(MyFFTCPUType *)fftw_malloc(sizeof(MyFFTCPUType)*maxThreads)); 
  #endif
  for(k=0;k<maxThreads;k++)
  {
    #ifdef SIMPLE
      planFFT[k]=fftwf_plan_dft_1d(fftSize, &xFFT[k*fftSize], &xFFT[k*fftSize], FFTW_FORWARD, planType);
    #else
      planFFT[k]= fftw_plan_dft_1d(fftSize, &xFFT[k*fftSize], &xFFT[k*fftSize], FFTW_FORWARD, planType);
    #endif
  }

  #ifdef OMP
    #pragma omp parallel
  #endif
  {
    int myID, myIDpos, pos, i, j;

    #ifdef OMP
      myID=omp_get_thread_num();
    #else
      myID=0;
    #endif
    myIDpos=myID*fftSize;

    #ifdef OMP
      #pragma omp for
    #endif
    for(i=0; i<nFrames; i++)
    {
      pos=i*(winSize-NoOverlap);
      
      #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
      for(j=0; j<winSize; j++) { xFFT[myIDpos+j]=audio[pos+j] * vHanning[j]; }

      /* We need this for the cases where winsize < fftSize. We observed */
      /* issues with 7 seconds audio ¿why not with long audios? ?¿?¿     */
      if(winSize<fftSize)
        memset(&xFFT[myIDpos+winSize], 0, sizeof(MyFFTcompl)*(fftSize-winSize));

      #ifdef SIMPLE
        fftwf_execute(planFFT[myID]);
      #else
        fftw_execute(planFFT[myID]);
      #endif

      pos=i*rowsNMF;
      
      #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
      for(j=0; j<rowsNMF; j++)
      {
        #ifdef SIMPLE
          sNMF[pos+j]=cabsf(xFFT[myIDpos+j]);
        #else
          sNMF[pos+j]= cabs(xFFT[myIDpos+j]);
        #endif
      }
    }
  }

  for(k=0;k<maxThreads;k++)
  {
    #ifdef SIMPLE
      fftwf_destroy_plan(planFFT[k]);
    #else
      fftw_destroy_plan(planFFT[k]);
    #endif
  }
  
  #ifdef SIMPLE
    fftwf_free(xFFT);
    fftwf_free(planFFT);
  #else
    fftw_free(xFFT);
    fftw_free(planFFT);
  #endif

  return OK;
}


void randNMF(const int n, MyType *v)
{
  int i;

  MyType max=(MyType)RAND_MAX;
   
  #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
  for(i=0; i<n; i++)
    #ifdef SIMPLE
      v[i]=((MyType)rand() / max) + sEPS;
    #else
      v[i]=((MyType)rand() / max) + dEPS;
    #endif
}



/* Mersenne twister version "ar", generador de números pseudoaleatorios. */
/* Desarrollado en 1997 por Makoto Matsumoto y Takuji Nishimura. Hay una */
/* version para 64 bits (mt19937ar_64.c) que no usamos para comparar los */
/* los resultados que obtiene la version C con los de la version matlab  */
void randNMFmt19937ar(const int n, MyType *v)
{
  int i;

  init_genrand(13); // Hay que hablarlo siempre para que salga como en matlab 

  /* genrand_real3() generates a random double number on (0,1)  */
  #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
  for(i=0; i<n; i++)
    #ifdef SIMPLE
      v[i]=(float)genrand_real3() + 1.0;  // Peligroso
    #else
      v[i]=genrand_real3() + 1.0;
    #endif
}


/* Normaliza a 1, por columnas, los elementos de W, de dimensiones m (filas) n (columnas), y */
/* de H de dimensiones n (filas) y k (columnas). Las matrices estan almacenadas por columnas */
void normNMF(const int F, const int bc, const int T, MyType *W, MyType *H)
{
  int    i;
  MyType dtmp;
  
  for(i=0; i<bc; i++)
  {
    #ifdef SIMPLE
      dtmp=cblas_snrm2(F, &W[i*F], 1);
    
      cblas_sscal(F, 1.0/dtmp, &W[i*F], 1);
      cblas_sscal(T,     dtmp, &H[i],  bc);
    #else
      dtmp=cblas_dnrm2(F, &W[i*F], 1);
    
      cblas_dscal(F, 1.0/dtmp, &W[i*F], 1);
      cblas_dscal(T,     dtmp, &H[i],  bc);
    #endif
  }
} 


/* Actualmente no se usa. Se hace la suma de eps cuando se usa en vector */
void epsNMF(const int n, MyType *v)
{
  int i;

  #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v"
  for(i=0; i<n; i++)
    #ifdef SIMPLE
      v[i]=v[i]+sEPS;
    #else
      v[i]=v[i]+dEPS;
    #endif
}


/* El core computacional de esta funcion es la operacion:                      */
/*      dtmp = sNMF[i] * (logf(sNMF[i]) - logf(yNMF[i]) + yNMF[i] + sNMF[i])   */
/* Que calcula muchos LOG, que son lentos. Habria que buscar alternativas fast */
/* Cambiado LOG(a)-LOG(b) por LOG(a/b), que en nuestro caso no tiene problemas */
/* de estabilidad. En secuencial, usando LOG(a)-LOG(b), distNMF era el 40% del */
/* tiempo de uNMF. Ahora es el 25% del tiempo. En paralelo "acelera" bien      */
MyType distNMF(const int F, const int T, const int bc, MyType *sNMF, MyType *yNMF, MyType *hNMF)
{
  int i;
  MyType dtmp, dist=0.0;
  
  #ifndef LAPACKE  
    int j, pos;
    #ifdef SIMPLE
      float  dsum=FLT_MIN;
    #else
      double dsum=DBL_MIN;
    #endif
  #endif

  #ifdef OMP
    #pragma omp parallel for reduction(+:dist) private(dtmp)
  #endif
  for(i=0; i<F*T; i++)
  {
    #ifdef SIMPLE
      dtmp = sNMF[i] * logf(sNMF[i]/yNMF[i]) + yNMF[i] - sNMF[i];
    #else
      dtmp = sNMF[i] *  log(sNMF[i]/yNMF[i]) + yNMF[i] - sNMF[i];
    #endif
    dist+=dtmp*dtmp;
  }

  #ifdef LAPACKE
    #ifdef SIMPLE
      dist=sqrtf(dist) + lambda*LAPACKE_slange(CblasColMajor,'1', bc, T, hNMF, bc);
    #else
      dist=sqrt(dist)  + lambda*LAPACKE_dlange(CblasColMajor,'1', bc, T, hNMF, bc);
    #endif
    return dist;
  #else
    #ifdef OMP
      #pragma omp parallel for private(dtmp, pos, j) reduction(max: dsum) 
    #endif
    for(i=0; i<T; i++)
    {
      dtmp=0.0; pos=i*bc;
      for(j=0; j<bc; j++)
        dtmp += hNMF[pos+j];
      if (dtmp > dsum) { dsum=dtmp; }
    }
    return sqrt(dist)+lambda*dsum;
  #endif
}  


/* La NMF no supervisada. No mas comentarios */
int uNMF(const int F, const int T, const int bc, MyType *sNMF, MyType *wNMF, MyType *hNMF, MyType *yNMF, int *rept)
{
  #ifdef DEPURA
    FILE *fdepura;
  #endif

  int
    i, j, k, idx, endNMF=0;
    
  MyType
    *vONES=NULL, *vDeno=NULL, *mTemp, xlambda=lambda*((MyType)F/(MyType)bc);

  #ifdef SIMPLE
    float  dklpre=FLT_MAX, dklnow;
  #else
    double dklpre=DBL_MAX, dklnow;
  #endif


  /* Esto es una putada pero es necesario. Se puede arreglar */
  CHECKNULL(vONES=(MyType *)malloc(max(F,T)   * sizeof(MyType)));
  CHECKNULL(vDeno=(MyType *)malloc(bc         * sizeof(MyType)));
  CHECKNULL(mTemp=(MyType *)malloc(bc*max(F,T)* sizeof(MyType)));

  memSetValue(max(F,T), 1.0, vONES);

  /* Random wNMF and hNMF. Por ahora aqui usamos la generación del */
  /* C para poder comparar con matlab, cuando todo Ok se unificara */
  //randNMFmt19937ar(F*bc, wNMF);
  //randNMFmt19937ar(bc*T, hNMF);
  randNMF(F*bc, wNMF);
  randNMF(bc*T, hNMF);


  /* Normalizations */
  /* Matlab: Fc=sqrt(sum(W.^2)); W=W./repmat(Fc,F,1); H=(repmat(Fc,T,1)').*H;  */
  /* Calcula la sqrt de la suma al cuadrado de los elementos de W por columnas */
  /* El resultado es un vector de bc elementos (bc=columnas de W y filas de H) */
  /* Luego divide cada elemento de W por la suma correspondiente a su columna  */
  /* Lo mismo para H pero como es 1D-colum en H el stride no es 1 es bc        */
  normNMF(F, bc, T, wNMF, hNMF);

  /* yNMF inicialization.  */
  /* Matlab: Y=W*H+eps; */
  #ifdef SIMPLE
    cblas_sgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, F, T, bc, 1.0, wNMF, F, hNMF, bc, 0.0, yNMF, F);
  #else
    cblas_dgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, F, T, bc, 1.0, wNMF, F, hNMF, bc, 0.0, yNMF, F);
  #endif
  // epsNMF(F*T, yNMF); /* No necesario. Antes de usar yNMF se le vuelve a sumar un eps */


  i=0;
  while ((i<maxIter) && (!endNMF))
  {
    /* ******************************************************************* */
    /*                              INICIO Paso 1                          */
    /* ******************************************************************* */
    /* Matlab: W=W.*(((Y.^(beta-2)+eps).*S)*H') ./ ((Y.^(beta-1))*H'+eps); */
    /* Beta vale siempre 1, por tanto es lo mismo que hacer lo siguiente   */
    /* W=W .* (((Y.^-1+eps).*S)*H') ./ ((Y.^0)*H'+eps);                    */
    
    /* (Y.^0)=matriz todo 1's (la llamamos ONES). ONES*H'=(H*ONES')'. El   */
    /* producto matricial (H*ONES')' es lo mismo que hacer H por un vector */
    /* de unos, se le suma el eps y luego se usa en la operacion ./        */
    /* Es decir, cambiamos el producto matricial por uno matriz x vector   */
    /* Se hace la suma del eps y entonces vDeno equivale a (Y.^0)*H'+eps   */
    #ifdef SIMPLE
      cblas_sgemv(CblasColMajor, CblasNoTrans, bc, T, 1.0, hNMF, bc, vONES, 1, 0.0, vDeno, 1);
    #else
      cblas_dgemv(CblasColMajor, CblasNoTrans, bc, T, 1.0, hNMF, bc, vONES, 1, 0.0, vDeno, 1);
    #endif
    // epsNMF(bc, vDeno);  /* Sumamos eps directamente cuando se use vDeno */

    /* (((Y.^-1+eps).*S)*H'). (Y.^-1+eps)=(1/Y + eps). Luego la operacion  */
    /* ((Y.^-1+eps).*S) es dividir los elementos de S por los de (Y+eps)   */
    /* Dejamos el resultado en la propia Y ya que vDeno ya calculado.      */
    /* Necesita OpenMP >= 4.0. Gcc 4.8 no soporta Openmp < 4.0             */
    #ifdef OMP
      #pragma omp parallel for simd
    #endif
    for(j=0; j<F*T; j++) 
    #ifdef SIMPLE
      yNMF[j]=sNMF[j] / (yNMF[j] +sEPS);
    #else
      yNMF[j]=sNMF[j] / (yNMF[j] +dEPS);
    #endif

    /* Finalmente hacemos Y*H'. Necesita almacenamiento temporal Sea mTemp */
    #ifdef SIMPLE
      cblas_sgemm(CblasColMajor, CblasNoTrans, CblasTrans, F, bc, T, 1.0, yNMF, F, hNMF, bc, 0.0, mTemp, F);
    #else
      cblas_dgemm(CblasColMajor, CblasNoTrans, CblasTrans, F, bc, T, 1.0, yNMF, F, hNMF, bc, 0.0, mTemp, F);
   #endif

    /* En estos momentos tenemos W=(W .* mTemp)./vDeno. Se hace en un paso */
    /* Podria ser for(j=0;j<F*bc;j++) wNMF[j]=wNMF[j]*mTemp[j]/vDeno[j%F]; */
    /* Pero la operacion % no es desable por ¿rendimiento para el for?     */
    #ifdef OMP
      #pragma omp parallel for private(idx, k)
    #endif
    for(j=0; j<bc; j++)
    {
      idx=j*F;
      #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v" o "#pragma omp simd"
      for(k=0; k<F; k++)
      {
        #ifdef SIMPLE
          wNMF[idx+k] = wNMF[idx+k] * mTemp[idx+k] / (vDeno[j]+sEPS);
        #else
          wNMF[idx+k] = wNMF[idx+k] * mTemp[idx+k] / (vDeno[j]+dEPS);
        #endif
        //if (wNMF[idx+k]<0.0) wNMF[idx+k]=0.0; /* NO procede debe ser >=0 */
      }
    }
    /* ******************************************************************* */
    /*                                FIN Paso 1                           */
    /* ******************************************************************* */


    /* Repetir el proceso de normalizacion de W y H y obtener Y            */
    normNMF(F, bc, T, wNMF, hNMF);
    #ifdef SIMPLE
      cblas_sgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, F, T, bc, 1.0, wNMF, F, hNMF, bc, 0.0, yNMF, F);
    #else
      cblas_dgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, F, T, bc, 1.0, wNMF, F, hNMF, bc, 0.0, yNMF, F);
    #endif
    // epsNMF(F*T, yNMF); /* No necesario */
    
    
    /* ******************************************************************* */    
    /*                              INICIO Paso 2                          */
    /* ******************************************************************* */
    /* Recordar que beta=1: Y1.^(beta-2)=(Y1.^-1) e Y.^(beta-1)=Y.^0       */
    /* Matlab: H=(H.*(W'*((Y.^-1+eps).*S))) ./ (W'*(Y.^0)+lambda*(F/bc));  */

    /* W'*(Y.^0)+lambda*(F/bc). lambda, F y bc son constantes. Se calcula  */
    /* una vez al inicio y se llaman xlambda=lambda*(F/bc). (Y.^0) es una  */
    /* matriz de unos. Volvemos a cambiar el producto matricial W'*(Y.^0)  */
    /* por matrizxvector. Necesitamos un vector de tamano F de unos. Lo    */
    /* hemos llamado vONES y ya ha sido utilizado en el Primer paso. El    */
    /* resultado es un vector, de tamano F, que ya existe del Primer paso  */
    #ifdef SIMPLE
      cblas_sgemv(CblasColMajor, CblasTrans, F, bc, 1.0, wNMF, F, vONES, 1, 0.0, vDeno, 1);
    #else
      cblas_dgemv(CblasColMajor, CblasTrans, F, bc, 1.0, wNMF, F, vONES, 1, 0.0, vDeno, 1);
    #endif
    #ifdef OMP
      #pragma omp parallel for simd
    #endif
    for(j=0; j<bc; j++) { vDeno[j] = vDeno[j]+xlambda; }

    /* (W'*((Y.^-1+eps).*S)). Primero ((Y.^-1+eps).*S), que es dividir los */
    /* elementos de S por los de Y+eps. Ya lo hemos hecho en Primer Paso.  */
    /* Necesita OpenMP >= 4.0. Gcc 4.8 no soporta Openmp < 4.0             */
    #ifdef OMP
      #pragma omp parallel for simd
    #endif
    for(j=0; j<F*T; j++) 
    #ifdef SIMPLE
      yNMF[j]=sNMF[j] / (yNMF[j] +sEPS);
    #else
      yNMF[j]=sNMF[j] / (yNMF[j] +dEPS);
    #endif

    /* Finalmente hacemos W'*Y. Necesita almacenamiento temporal, como en  */
    /* Primer Paso. Usamos el ya creado  mTemp                             */
    #ifdef SIMPLE
      cblas_sgemm(CblasColMajor, CblasTrans, CblasNoTrans, bc, T, F, 1.0, wNMF, F, yNMF, F, 0.0, mTemp, bc);
    #else
      cblas_dgemm(CblasColMajor, CblasTrans, CblasNoTrans, bc, T, F, 1.0, wNMF, F, yNMF, F, 0.0, mTemp, bc);
    #endif
    
    /* En estos momentos tenemos H=(H.*mTemp) ./ vDeno. Se hace en un paso */
    /* Podria ser for(j=0;j<T*bc;j++) hNMF[j]=hNMF[j]*mTemp[j]/vDeno[j%bc] */
    /* Pero la operacion % no es desable por ¿rendimiento para el for?     */
    #ifdef OMP
      #pragma omp parallel for private(idx, k)
    #endif
    for(j=0; j<T; j++)
    {
      idx=j*bc;
      #pragma GCC ivdep // por claridad mejor que "MyType *__restrict v" o "#pragma omp simd"
      for(k=0; k<bc; k++)
      {
        hNMF[idx+k] = (hNMF[idx+k] * mTemp[idx+k]) / vDeno[k];
        //if (hNMF[idx+k]<0.0) hNMF[idx+k]=0.0; /* NO procede debe ser >=0 */
      }
    }
    /* ******************************************************************* */
    /*                                FIN Paso 2                           */
    /* ******************************************************************* */


    /* Terminamos actualizando la Y                                        */
    #ifdef SIMPLE
      cblas_sgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, F, T, bc, 1.0, wNMF, F, hNMF, bc, 0.0, yNMF, F);
    #else
      cblas_dgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, F, T, bc, 1.0, wNMF, F, hNMF, bc, 0.0, yNMF, F);
    #endif
    // epsNMF(F*T, yNMF); /* No lo veo necesario */


    /* ********************************************************************* */
    /*                              Criterio Parada                          */
    /* ********************************************************************* */    
    /* Ver comentarios en la funcion distNMF sobre el uso de los LOG         */
    dklnow=distNMF(F, T, bc, sNMF, yNMF, hNMF);
    
    if ((dklpre-dklnow) < (dklnow*eta)) { endNMF=1; } else { dklpre=dklnow; }

    i++;
  }
  *rept=i;

  #ifdef DEPURA
    CHECKNULL(fdepura=fopen("wNMF.dat", "wb"));
    fwrite(&F,   sizeof(int),       1, fdepura);
    fwrite(&bc,  sizeof(int),       1, fdepura);
    fwrite(wNMF, sizeof(MyType), F*bc, fdepura);
    CHECKERR(fclose(fdepura));

    CHECKNULL(fdepura=fopen("hNMF.dat", "wb"));
    fwrite(&bc,  sizeof(int),       1, fdepura);
    fwrite(&T,   sizeof(int),       1, fdepura);
    fwrite(hNMF, sizeof(MyType), bc*T, fdepura);
    CHECKERR(fclose(fdepura));
  #endif

  free(vONES);  free(vDeno);  free(mTemp);
  return OK;
}


/* Algoritmo de estimación del ritmo cardiaco. BIBLIOGRAFIA:                                             */
/* [1] Study and Design of a Shannon-Energy-Envelope based Phonocardiogram Peak Spacing Analysis for     */
/*     Estimating Arrhythmic Heart-Beat                                                                  */
/* [2] Power-Scaled Spectral Flux and Peak-Valley Group-Delay Methods for Robust Musical Onset Detection */
/* [3] An Efficient Algorithm for Automatic Peak Detection in Noisy Periodic and Quasi-Periodic Signal   */
/* [4] Heart Rate Variability using Shannon Energy                                                       */
int HSD(const int nSamples, const int winSMst, const MyType *audio, const MyType *TsMst, MyType **Times,
        int *Patrones, const int maxThreads, const int planType)
{
  int          winSize, fftSize, CxSize, NoOverlap, NoOverMst, nFrames, nFraMst, nFrecCP, *Peaks=NULL;
  int          i, j, pos1, pos2, idx1=0, idx2=0;
  MyType       *vHanning=NULL, *sNMF=NULL, *Ts=NULL, *R=NULL, *Cx=NULL;
  MyType       max1, max2, dtmp, perEsti;

  winSize  = freqHz*secondsHSD;
  fftSize  = nextPow2(winSize);
  nFrecCP  = ceil((freqCut*fftSize)/freqHz); 
  NoOverlap= round(fftSize*overlap);  
  NoOverMst= round(nextPow2(winSMst)*overlap);
  nFrames  = floor((nSamples-NoOverlap)/(winSize - NoOverlap));
  nFraMst  = floor((nSamples-NoOverMst)/(winSMst - NoOverMst));
  CxSize   = 2*nFrames-1;
  
  CHECKNULL(vHanning=(MyType *)calloc(winSize,         sizeof(MyType)));
  CHECKNULL(      Ts=(MyType *)calloc(nFrames,         sizeof(MyType)));
  CHECKNULL(       R=(MyType *)calloc(nFrames,         sizeof(MyType)));
  CHECKNULL(   Peaks=(int    *)calloc(nFrames,         sizeof(int)));
  CHECKNULL(      Cx=(MyType *)calloc(CxSize,          sizeof(MyType)));
  CHECKNULL(    sNMF=(MyType *)calloc(nFrecCP*nFrames, sizeof(MyType)));  

  CHECKERR(hanning(winSize, vHanning));
  CHECKERR(rHannFFT(winSize, nFrames, NoOverlap, audio, vHanning, nFrecCP, sNMF, fftSize, maxThreads, planType));

  /* Construir funcion onset basada en los cambios de energía (Spectral flux). Obtiene R(i)>=0 */
  pos1=nFrecCP;
  #ifdef OMP2
    #pragma omp parallel for private(pos2, dtmp, j) firstprivate(pos1) // ¿overhead > saved time.?
  #endif
  for(i=1;i<nFrames;i++)
  {
    pos2=pos1-nFrecCP;  dtmp=0.0;
    for(j=0; j<nFrecCP; j++)
      dtmp+=sNMF[pos1+j]-sNMF[pos2+j];

    /* Si dtmp<0.0 entonces R[i]=0.0, otro caso R[i]=dtmp. ¿Mejor esto o un if? */
    #ifdef SIMPLE
      R[i]=(dtmp+fabsf(dtmp))/2.0;
    #else
      R[i]=(dtmp+ fabs(dtmp))/2.0;
    #endif    
    pos1+=nFrecCP;
  }

  /* Calculamos la autocorrelacion para estimar el periodo de la señal de entrada */
  xcorr(R, Cx, NULL, nFrames, maxThreads);

  /* Uso del algoritmo AMPD Automatic Multiscale-based Peak Detection doi:10.3390/a5040588 */
  CHECKERR(AMbPD(nFrames, &Cx[nFrames], Peaks));

  max1=Cx[nFrames+Peaks[1]];
  #ifdef OMP
    #pragma omp parallel for reduction(max: max1)
  #endif
  for(i=nFrames+Peaks[1]+1;i<CxSize;i++)
    if (Cx[i]>max1) max1=Cx[i];
  idx1=buscarFirstIgual(nFrames, &Cx[nFrames], max1);

  max2=Cx[nFrames+idx1+1];
  #ifdef OMP
    #pragma omp parallel for reduction(max: max2)
  #endif
  for(i=nFrames+idx1+2;i<CxSize;i++)
    if (Cx[i]>max2) max2=Cx[i];
  idx2=buscarFirstIgual(nFrames, &Cx[nFrames], max2);

  /* Building vector T */
  dtmp=(MyType)(winSize-NoOverlap)/(MyType)freqHz;
  #ifdef OMP2
    #pragma omp parallel for simd  // ¿overhead > saved time.?
  #else
    #pragma GCC ivdep
  #endif
  for(i=0;i<nFrames;i++)
    Ts[i]=i*dtmp;

  /* Asegura que los distancia entre sonidos iguales es mayor que un valor lógico de un latido de corazón */
  if ((Ts[idx2]>Ts[idx1]) && (Ts[idx2]<1.0) && ((Ts[idx2]-Ts[idx1])>0.4))
    perEsti=Ts[idx2+1];
  else
    perEsti=Ts[idx1+1];

  /* Uso del algoritmo AMPD Automatic Multiscale-based Peak Detection doi:10.3390/a5040588 */
  CHECKERR(AMbPD(nFrames, R, Peaks));  

  /* Times los tiempos donde estan los sonidos de corazon  */
  CHECKNULL((*Times)=(MyType *)calloc(Peaks[0]+2, sizeof(MyType)));  

  max1=Ts[Peaks[1]]-perEsti;
  if (max1 > 0)
  {
    (*Times)[1]=max1;
    for(i=1; i<=Peaks[0]; i++)
      (*Times)[i+1]=Ts[Peaks[i]];
    (*Times)[0]=Peaks[0]+1;
    idx1=buscarFirstMayorIgual(nFrames, Ts, max1);
    memmove(&Peaks[2], &Peaks[1], sizeof(int)*Peaks[0]);
    Peaks[0]++;
    Peaks[1]=idx1-1;
  }
  else
  {
    max1=Ts[Peaks[2]]-perEsti;
    if ((max1 > 0) && (max1<Ts[Peaks[1]+1]))
    {
      (*Times)[1]=max1;
      for(i=1; i<=Peaks[0]; i++)
        (*Times)[i+1]=Ts[Peaks[i]];
      (*Times)[0]=Peaks[0]+1;
      idx1=buscarFirstMayor(nFrames, Ts, max1);
      memmove(&Peaks[2], &Peaks[1], sizeof(int)*Peaks[0]);
      Peaks[0]++;
      Peaks[1]=idx1-1;
    }
    else
    {
      for(i=1; i<=Peaks[0]; i++)
        (*Times)[i]=Ts[Peaks[i]];
      (*Times)[0]=Peaks[0];
    }
  }  

  #ifdef OMP2
    #pragma omp parallel for private(idx1, idx2, pos1)
  #endif
  for(i=1;i<=Peaks[0];i++)
  {
    /* Ultima posicion de Ts del Master (TsMst) de valor menor o igual que Ts[Peaks[i]] */
    /* Si se compara con matlab en el audio "artificial" de 70 segundos hay un problema */
    /* de estabilidad (redondeos) que hace que un par de intervalos sean distintos. Si  */
    /* se quiere comparar usar "Ts[Peaks[i]]+0.1e-13" en vez de "Ts[Peaks[i]]"          */
    idx1=buscarLastMenorIgual(nFraMst, TsMst, Ts[Peaks[i]]); /* Ts[Peaks[i]]+0.1e-13);  */

    if (idx1<0) { idx1=2; idx2=0; } else { idx2=idx1+1; }

    switch (winSMst)
    {
      case  512:
        if(idx1>0) { idx1=idx1-1; idx2=idx2+2; } else { idx1=0; idx2=idx2+1; }
        break;
      case 1024:
        if(idx1>0) { idx1=idx1-1; idx2=idx2+1; } else { idx1=0; idx2=idx2+1; }
        break;
      case 2048:
        if(idx1>0) { idx1=idx1-1;              } else { idx1=0; }
        break;
      default:
        if(idx1>0) { idx1=idx1-1; idx2=idx2+2; } else { idx1=0; idx2=idx2+1; }
    }
    
    pos1=idx1;
    while(pos1<=idx2)
    {
      Patrones[pos1]=1;
      pos1++;
    }
  }
  free(vHanning); free(sNMF); free(Ts); free(R); free(Peaks); free(Cx);
  
  return OK;
}


/* Implementacion en C del algoritmo AMPD Automatic Multiscale-based Peak Detection doi:10.3390/a5040588 */
int AMbPD(const int n, const MyType *R, int *Peaks)
{
  int    i, k, m, count;
  MyType *LSM=NULL, *vONES=NULL, *G=NULL;

  m=ceil(n/2.0)-1;

  CHECKNULL(  LSM=(MyType *)calloc(m*n, sizeof(MyType)));
  CHECKNULL(vONES=(MyType *)malloc(n  * sizeof(MyType)));
  CHECKNULL(    G=(MyType *)calloc(n  , sizeof(MyType)));

  memSetValue(n, 1.0, vONES);
  randNMFmt19937ar(m*n, LSM);

  for(k=0; k<m; k++)
    for(i=k+1; i<n-k-1; i++)
      if ((R[i]>R[i-1-k]) && (R[i]>R[i+1+k])) LSM[k+(i+1)*m]=0.0;

  #ifdef SIMPLE
    cblas_sgemv(CblasColMajor, CblasNoTrans, m, n, 1.0, LSM, m, vONES, 1, 0.0, G, 1);
    k=max(cblas_isamin(m, G, 1)+1, 2);
  #else
    cblas_dgemv(CblasColMajor, CblasNoTrans, m, n, 1.0, LSM, m, vONES, 1, 0.0, G, 1);
    k=max(cblas_idamin(m, G, 1)+1, 2);
  #endif

  count=0;
  #ifdef OMP3
    #pragma omp parallel
  #endif
  {
    int pos, j;
    MyType   dtmp, dtmp1, media;
    #ifdef OMP3
      #pragma omp for
    #endif
    for(i=0;i<n;i++)
    {
      dtmp=0.0;
      pos =i*m;
      for(j=0;j<k;j++)
        dtmp+=LSM[pos+j]; 
      media=dtmp/k;

      dtmp =0.0;    
      for(j=0;j<k;j++)
      {
        dtmp1=LSM[pos+j]-media;
        dtmp+=dtmp1*dtmp1;
      }

      if (sqrt(dtmp/(k-1.0))==0)
      {
        #ifdef OMP3
          #pragma omp critical (contador)
        #endif
        {
          count++;
          Peaks[count]=i-1;
        }
      }
    }
  }
  Peaks[0]=count; 
  #ifdef OMP3
    qsort(&Peaks[1], count, sizeof(int), compara);
  #endif


  free(LSM); free(vONES); free(G);
  
  return OK;
}


int invSpectroExtended(const int N, const int nFrames, const int bc, const int winSize, const int M,
                       const int NoOverlap, const MyType *vHanning, const MyType *wNMF, const MyType *hNMF,
                       MyType *Mat, MyFFTcompl *dest, MyFFTcompl *xIFFT, MyFFTCPUType planIFFT)
{
  int i, j, start, pos, stride, stridePos;
  MyType Z=(MyType)M;

  /* N=rowsNMF      M=fftSize    Z=(Mytype)fftSize */

  /* Primer frame, es distinto del resto */
  Mat[0]  =xIFFT[0]  =wNMF[0]  *hNMF[0];
  Mat[N-1]=xIFFT[N-1]=wNMF[N-1]*hNMF[0];
  
  #pragma GCC ivdep
  for(j=1;j<N-1;j++) 
  {
    Mat[j] = xIFFT[j] = xIFFT[M-j] = wNMF[j]*hNMF[0];
  }

  #ifdef SIMPLE
    fftwf_execute(planIFFT);
  #else
    fftw_execute(planIFFT);
  #endif

  /* FFTW computes an unnormalized transform: computing an FFT followed by an IFFT  */
  /* transform (or vice versa) will result in the original data multiplied by the   */
  /* size of the transform (the product of the logical dimensions). Matlab computes */
  /* normalized transform ?¿?¿. Then we need to divide by fftSize for equal results */
  #pragma GCC ivdep
  for(j=0; j<winSize;j++) 
    dest[j]=creal(xIFFT[j])/Z;


  /* Resto de Frames */
  stride=winSize-NoOverlap; pos=bc; start=N;
  for(i=1;i<nFrames;i++)
  {
    Mat[start]    =xIFFT[0]  =wNMF[0]  *hNMF[pos];
    Mat[start+N-1]=xIFFT[N-1]=wNMF[N-1]*hNMF[pos];

    #pragma GCC ivdep
    for(j=1;j<N-1;j++)
    {
      Mat[start+j] = xIFFT[j] = xIFFT[M-j] = wNMF[j]*hNMF[pos];
    }

    #ifdef SIMPLE
      fftwf_execute(planIFFT);
    #else
      fftw_execute(planIFFT);
    #endif

    stridePos=i*stride;
    
    #pragma GCC ivdep
    for(j=0; j<winSize/2;j++)
      dest[stridePos+j]=(creal(dest[stridePos+j])+creal(xIFFT[j])/Z) / (vHanning[j]+vHanning[stride+j]);
  
    #pragma GCC ivdep  
    for(j=winSize/2;j<winSize;j++)
      dest[stridePos+j]=creal(xIFFT[j])/Z;

    pos  +=bc;
    start+=N;
  }
  
  return 0;
}


int invSpectro(const int N, const int nFrames, const int winSize, const int M, const int NoOverlap,
               const MyType *vHanning, const MyFFTcompl *src, MyFFTcompl *dest, MyFFTcompl *xIFFT, MyFFTCPUType planIFFT)
{
  int    i, j, start, stride, pos;
  MyType Z=(MyType)M;

  /* N=rowsNMF      M=fftSize    Z=(Mytype)fftSize */

  /* Primer frame, es distinto del resto */
  xIFFT[0]  =src[0];
  xIFFT[N-1]=src[N-1];

  #pragma GCC ivdep
  for(j=1;j<N-1;j++)
  { 
    xIFFT[j]  =src[j];
    xIFFT[M-j]=conj(src[j]);
  }

  #ifdef SIMPLE
    fftwf_execute(planIFFT);
  #else
    fftw_execute(planIFFT);
  #endif


  /* FFTW computes an unnormalized transform: computing an FFT followed by an IFFT  */
  /* transform (or vice versa) will result in the original data multiplied by the   */
  /* size of the transform (the product of the logical dimensions). Matlab computes */
  /* normalized transform ?¿?¿. Then we need to divide by fftSize for equal results */
  #pragma GCC ivdep
  for(j=0; j<winSize;j++)
    dest[j]=creal(xIFFT[j])/Z;


  /* Resto de Frames */
  stride=winSize-NoOverlap; start=N;
  for(i=1;i<nFrames;i++)
  {
    xIFFT[0]  =src[start];
    xIFFT[N-1]=src[N+start-1];

    #pragma GCC ivdep
    for(j=1;j<N-1;j++)
    {
      xIFFT[j]  =src[start+j];
      xIFFT[M-j]=conj(src[start+j]);
    }

    #ifdef SIMPLE
      fftwf_execute(planIFFT);
    #else
      fftw_execute(planIFFT);
    #endif

    pos=i*stride;
    
    #pragma GCC ivdep  
    for(j=0; j<winSize/2;j++)
      dest[pos+j]=(creal(dest[pos+j])+creal(xIFFT[j])/Z) / (vHanning[j]+vHanning[stride+j]);
  
    #pragma GCC ivdep  
    for(j=winSize/2;j<winSize;j++)
      dest[pos+j]=creal(xIFFT[j])/Z;

    start+=N;
  }
  
  return 0;
}


/* Correlacion entre la matriz de activaciones obtenidas en la descomposicion NMF */
/* y el patron generado. Preprocesa el valor de la matriz de activaciones H(i,:), */
/* para que sus valores se encuentren en el mismo rango que los del patron, [0,1] */
/*                                                                                */
/*                                                                                */
/* Usa el coeficiente de correlación de Pearson (cP). cP(x,y) se calcula como:    */
/*           cP(x,y)=cov(x,y) / (des(x)*des(y))                                   */
/* El calculo de la covarianza de x e y (cov(x,y)) se calcula  como:              */
/*           cov(x,y)=(1/(n-1)) * (Sumatorio(x(i)y(i)) - media(x)*media(y))       */
/* La desviacion estandar (des(x)) se calcula como:                               */
/*           des(x)=RAIZ((1/(n-1)) * Sumatorio(x(i)-media(x))^2)                  */
/* Y la media (media(x)) se calcula como                                          */
/*            media(x)=(1/N) * Sumatorio(x(i))                                    */
/*                                                                                */
/*                                                                                */
/* Variable Y es Patrones. Variable X=H/maximo(H) con H la fila en curso de H, no */
/* la matriz H completa, un vector. Como el uso de X es para saber si cada X(i)es */
/* mayor o menor que la media de X, podemos evitar calcular el maximo. Esto es,   */
/* en vez de                                                                      */
/*    X(i)=H(i)/max(H), media=(1/N)*Sumatorio(X(i)), "si X(i)>media xx else yy"   */
/* hacer                                                                          */
/*    X(i)=H(i), media=(1/N)*Sumatorio(X(i)), "si X(i)>media xx else yy"          */
/* Que debe dar lo mismo. ESTO EVITA hacer un bucle de tamano nFrames   al inicio */ 
/* Por otra parte, Patrones(i)=1 o Patrones(i)=0. No puede valer otra cosa        */
/* El codigo Matlab tiene lo siguente:                                            */
/*    aux1(aux1>mean(aux1))=1; que en lo nuestro es "si (X(i) > media(X)) X(i)=1" */
/*    aux1(aux1<mean(aux1))=0; que en lo nuestro es "si (X(i) < media(X)) X(i)=0" */
/* ¿Que pasa con los aux1==mean(aux1)? aux1(i) debe ser 0 o 1. Por ello usamos    */
/*    "si (X(i) > media(X)) X(i)=1 else X(i)=0"                                   */
MyType corrPatron(const int bc, const int nFrames, const MyType *hNMF, const int *Y, MyType *X)
{
  MyType CoVaria, dtmp, Xmean, Ymean, XdesStd, YdesStd;
  int    i;

  dtmp=0;
  for(i=0; i<nFrames;i++) { X[i]=hNMF[i*bc];  dtmp+=X[i]; }
  dtmp=dtmp/(MyType)nFrames;

  Xmean=Ymean=0;
  for(i=0; i<nFrames;i++)
  {
    Ymean+=Y[i];
    if (X[i] > dtmp) { X[i]=1; Xmean++; } else { X[i]=0; }
  }

  Xmean=Xmean/(MyType)nFrames; 
  Ymean=Ymean/(MyType)nFrames;

  XdesStd=YdesStd=CoVaria=0;
  dtmp=Xmean*Ymean;
  for(i=0;i<nFrames;i++)
  {
    XdesStd+=(X[i]-Xmean)*(X[i]-Xmean);
    YdesStd+=(Y[i]-Ymean)*(Y[i]-Ymean);
    CoVaria+=(MyType)(X[i]*Y[i]) - dtmp;
  }

  CoVaria/=(MyType)(nFrames-1);
  #ifdef SIMPLE
    XdesStd=sqrtf(XdesStd/(MyType)(nFrames-1));
    YdesStd=sqrtf(YdesStd/(MyType)(nFrames-1));
  #else
    XdesStd=sqrt(XdesStd/(MyType)(nFrames-1));
    YdesStd=sqrt(YdesStd/(MyType)(nFrames-1));
  #endif
 
  return CoVaria / (XdesStd*YdesStd);

}


/* Calculo de la frecuencia de roll-off */
MyType RollOff(const int n, const MyFFTcompl *X, const MyType threshold)
{
  int    i;
  MyType Energy=0, dtmp1, dtmp2;
  

  /* energia_total=frecuencia de Nyquist o fs/2. */
  for(i=0;i<(n/2);i++)
  {
    dtmp1  =cabs(X[i]);
    Energy+=dtmp1*dtmp1;
  }

  i=0;  dtmp1=0;
  while (dtmp1<=(Energy*threshold))
  {
    dtmp2 =cabs(X[i]);
    dtmp1+=dtmp2*dtmp2;
    i++;
  }

  return (((MyType)i*freqHz)/(MyType)n);
}


/* Calculo de la correlacion usando la distancia del coseno. Compara las bases entrenadas */
/* con las calculadas. Se compara cada W(:,i) con todas las columnas Wc(:,s). El valor de */ 
/* similitud para cada base i se obtiene como el maximo entre todos los valores obtenidos */
MyType corrCos(const int rowsNMF, const int colsBases, const MyType *W, const MyType *Bases, MyType *SIM)
{
  int    i;
  MyType dnorm2W, dtmp;

  #ifdef SIMPLE
    float  maxSIM=FLT_MIN;
  #else
    double maxSIM=DBL_MIN;
  #endif

  #ifdef SIMPLE
    cblas_sgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, 1, colsBases, rowsNMF, 1.0, W, 1, Bases, rowsNMF, 0.0, SIM, 1);
    dnorm2W=cblas_snrm2(rowsNMF, W, 1);
  #else
    cblas_dgemm(CblasColMajor, CblasNoTrans, CblasNoTrans, 1, colsBases, rowsNMF, 1.0, W, 1, Bases, rowsNMF, 0.0, SIM, 1);
    dnorm2W=cblas_dnrm2(rowsNMF, W, 1);
  #endif

  for(i=0;i<colsBases;i++)
  {
    #ifdef SIMPLE
      dtmp=SIM[i] / (dnorm2W*cblas_snrm2(rowsNMF, &Bases[i*rowsNMF], 1));
    #else
      dtmp=SIM[i] / (dnorm2W*cblas_dnrm2(rowsNMF, &Bases[i*rowsNMF], 1));
    #endif
    maxSIM=max(dtmp, maxSIM);
  }

  return maxSIM;
}


void ReconCorazon(const int n, MyFFTcompl *Xc, MyFFTcompl *Xp, const MyFFTcompl *cSNMF, const MyType *XcSpec, const MyType *XpSpec)
{
  int i;
  MyType dtmpc, dtmpp, dsum;

  #ifdef OMP
    #pragma omp parallel for private(dtmpc, dtmpp, dsum)
  #endif
  for(i=0; i<n;i++)
  {
    dtmpc=XcSpec[i]*XcSpec[i];  dtmpp=XpSpec[i]*XpSpec[i];  dsum=dtmpc+dtmpp;
    
    Xc[i]=(dtmpc/dsum)*cSNMF[i];
    Xp[i]=(dtmpp/dsum)*cSNMF[i];

  }
}


void linspace(const int left, const int right, const int n, MyType *v)
{
  int i;
  MyType stride;

  stride=(MyType)(right-left) / (MyType)(n-1);

  v[0]=(MyType)left; v[n-1]=(MyType)right;  

  for(i=1; i<n-1; i++) v[i]=v[i-1]+stride;
  
}


void interpl(const int N, const int M, const double *x, const double *y, const double *Vx, double *Vy)
{
  double xtmp;
  int low_i, low_ip1, high_i, mid_i, k;

  #pragma omp parallel for schedule(guided) private(low_i, low_ip1, high_i, mid_i, xtmp)
  for (k=0; k<M; k++)
  {
    if ((Vx[k] <= x[N-1]) && (x[0] <=Vx[k]))
    {
      low_i   = 1;
      low_ip1 = 2;
      high_i  = N;

      while (high_i > low_ip1)
      {
        mid_i = (low_i + high_i) >> 1;
        if (Vx[k] >= x[mid_i - 1]) { low_i = mid_i; low_ip1 = mid_i + 1; }  else { high_i = mid_i; }
      }

      xtmp = x[low_i - 1];
      xtmp = (Vx[k] - xtmp) / (x[low_i] - xtmp);

      if (xtmp == 0.0) {
        Vy[k] = y[low_i - 1];
      } else if (xtmp == 1.0) {
        Vy[k] = y[low_i];
      } else if (y[low_i - 1] == y[low_i]) {
        Vy[k] = y[low_i - 1];
      } else {
        Vy[k] = (1.0 - xtmp) * y[low_i - 1] + xtmp * y[low_i];
      }
    }
  }
}


int xcorr(double *X, double *Cx, double *Lags, const int Size, const int maxThreads)
{
  long         fftSize, x2SizeMinusOne, i, itmp;
  MyFFTcompl   *xFFT=NULL, *yFFT=NULL;
  MyFFTCPUType planFFT=NULL, planIFFT=NULL;

  x2SizeMinusOne=2*Size-1;
  fftSize=nextPow2(x2SizeMinusOne);

  #ifdef SIMPLE
    CHECKNULL(xFFT=(MyFFTcompl*)fftwf_malloc(sizeof(MyFFTcompl)*fftSize));

    #ifdef PARFFTW
      fftwf_plan_with_nthreads(maxThreads);
    #endif

    planFFT =fftwf_plan_dft_1d(fftSize, xFFT, xFFT, FFTW_FORWARD,  FFTW_ESTIMATE);
    planIFFT=fftwf_plan_dft_1d(fftSize, xFFT, xFFT, FFTW_BACKWARD, FFTW_ESTIMATE);
  #else
    CHECKNULL(xFFT=(MyFFTcompl*)fftw_malloc(sizeof(MyFFTcompl)*fftSize));
    CHECKNULL(yFFT=(MyFFTcompl*)fftw_malloc(sizeof(MyFFTcompl)*fftSize));
    
    #ifdef PARFFTW
      fftw_plan_with_nthreads(maxThreads);
    #endif
    
    planFFT =fftw_plan_dft_1d(fftSize, xFFT, xFFT, FFTW_FORWARD,  FFTW_ESTIMATE);
    planIFFT=fftw_plan_dft_1d(fftSize, xFFT, xFFT, FFTW_BACKWARD, FFTW_ESTIMATE);
  #endif

  for(i=0; i<Size; i++) { xFFT[i]=X[i]; }

  #ifdef SIMPLE
    fftwf_execute(planFFT);
  #else
    fftw_execute(planFFT);
  #endif


  /* El codigo matlab hace: a=d^2, seindo d=rt_hypotd_snf(parte.real, parte.imagen), que es d=cabs(complejo) */
  /* en resumen d=sqrt(parte.real^2 + parte.imagen^2). Luego a=d^2 es a=parte.real^2 + parte.imagen^2        */
  /* Como la FFTW no normaliza hay que hacerlo. Lo podemos hacer despues de la IFFT, diviendo por fftSize, o */
  /* aprovechamos ahora. Comprobado que la parte real, la que interesa, no cambia                            */
  #pragma omp parallel for simd
  for(i=0; i<fftSize; i++) 
    xFFT[i]=(creal(xFFT[i])*creal(xFFT[i]) + cimag(xFFT[i])*cimag(xFFT[i]))/(double)fftSize;

  #ifdef SIMPLE
    fftwf_execute(planIFFT);
  #else
    fftw_execute(planIFFT);
  #endif

  itmp=fftSize-Size+1;
  for(i=0;i<(Size-1);i++)
     Cx[i]=creal(xFFT[itmp+i]);

  itmp=Size-1;
  for(i=0;i<Size;i++)
    Cx[itmp+i]=creal(xFFT[i]);

  if(Lags != NULL)
    for(i=0; i<x2SizeMinusOne; i++) { Lags[i]=i - Size + 1; }

  #ifdef SIMPLE
    fftwf_free(xFFT);
    fftwf_destroy_plan(planFFT);
    fftwf_destroy_plan(planIFFT);
  #else
    fftw_free(xFFT);
    fftw_destroy_plan(planFFT);
    fftw_destroy_plan(planIFFT);
  #endif

  return OK;
}


MyType BMP(const int rowsNMF, const int nFrames, const int nSamples, MyFFTcompl *Xc, MyType *Audio, const int maxThreads)
{
  bool         Encontrado=false;
  int          i, j, tmp, T0_min_samp, T0_max_samp, cuantos;
  MyType       *x=NULL, *y=NULL, *xx=NULL, *yy=NULL, *Cx=NULL, dtmp;
  ValueAndPos  *Peaks=NULL;

  T0_max_samp=floor((60.0/BPM_min)*freqHz);
  T0_min_samp=floor((60.0/BPM_max)*freqHz);
  
  CHECKNULL(x    =(MyType      *)calloc(nFrames,      sizeof(MyType)));
  CHECKNULL(y    =(MyType      *)calloc(nFrames,      sizeof(MyType)));
  CHECKNULL(xx   =(MyType      *)calloc(nSamples,     sizeof(MyType)));
  CHECKNULL(yy   =(MyType      *)calloc(nSamples,     sizeof(MyType)));
  CHECKNULL(Cx   =(MyType      *)calloc(2*nSamples-1, sizeof(MyType)));
  CHECKNULL(Peaks=(ValueAndPos *)calloc(nSamples,     sizeof(ValueAndPos)));

  linspace(0, 1, nFrames,  x);
  linspace(0, 1, nSamples, xx);
  
  #pragma omp parallel for private(dtmp, j)
  for(i=0; i<nFrames; i++)
  {
    dtmp=0.0;
    for(j=0; j<rowsNMF; j++)
      dtmp += cabs(Xc[i*rowsNMF+j]);
    y[i]=dtmp;
  }

  interpl(nFrames, nSamples, x, y, xx, yy);

  /* Si queremos el vectro Lag tipo Matlab declararlo y pasarlo como 3 argumento */
double time=Ctimer();
  xcorr(yy, Cx, NULL, nSamples, maxThreads);

time=Ctimer()-time;
printf("EL tiempo de XcorrFFT es %f\n", time);


  tmp=floor((2*nSamples-1)/2);  
  for(i=0; i<=nSamples; i++)
    Cx[i]=Cx[i+tmp-1];

  cuantos=0;
  for(i=1; i<nSamples; i++)
    if ((Cx[i]>Cx[i-1]) && (Cx[i]>=Cx[i+1])) { Peaks[cuantos].data=Cx[i]; Peaks[cuantos].pos=i; cuantos++; }
  
  qsort(Peaks, cuantos, sizeof(ValueAndPos), comparadesc);

  i=0;
  while ((!Encontrado) && (i<cuantos))
  {
    if ((Peaks[i].pos > T0_min_samp) && (Peaks[i].pos < T0_max_samp)) 
      Encontrado=true;
    else
      i++;
  }

  #ifdef SIMPLE
    dtmp=roundf(60.0/(Peaks[i].pos/freqHz));
  #else
    dtmp=round (60.0/(Peaks[i].pos/freqHz));
  #endif

  free(x); free(y); free(xx); free(yy); free(Cx); free(Peaks);


  return dtmp;
}

